/************************************************************************
 * NAME:	jv.c
 *
 * DESCR:	Implements the JV1, JV3, & JVC floppy disk formats.
 *
 * NOTES:	
 ************************************************************************/

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>

#include "standard.h"
#include "floppy.h"
#include "error.h"
#include "jv.h"

/************************************************************************
 * NAME:	sector_map()
 *
 * DESCR:	Simple map to put the sector numbers in an order that
 *		would be more like a real floppy than linear.
 *		NOT USED!  I can imagine using this later to re-organize
 *			   the sectors to something more like a real disk.
 ************************************************************************/
static int sector_map[] = { 0, 5, 1, 6, 2, 7, 3, 8, 4, 9 };

/************************************************************************
 * NAME:	jv1_report()
 *
 * DESCR:	Generates a report of the (apparently) jv1 file.
 *
 * ARGS:	level refers to the amount of reporting, 1 is lowest,
 *		3 is highest.
 *
 * RETURNS:	
 *
 * NOTES:	- levels other than 1 cause a detailed look at the file
 *		  beyond the guessing that occurred before
 ************************************************************************/
int
jv1_report(int fd, struct floppy *floppy, int level)
{
    int	errors;

    printf("JV1");

    if (level == 1) {		/* first level, just indicate the type	*/
	printf("\n");
	return;
    }

    if (level == 2 || level == 3 || level == 4) {
	lseek(fd,(off_t)0,SEEK_SET);
	errors = jv1_read(fd,floppy);
	floppy_report(floppy,level,errors);
    }
}

/************************************************************************
 * NAME:	jv1_guesser()
 *
 * DESCR:	Tries to guess a jv1 format.  This is a very simple format
 *		so successful guessing has to look deep into the incoming
 *		file for a decent guess.   Since it is assuming TRS-80 disks
 *		it looks for "well-known" things such as (simply) file size
 *		first (this assumes that JV1 is SSSD simple 35 or 40 track
 *		diskettes).
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
jv1_guesser(int fd)
{
    long		size;
    int			i;
    int			histogram[256];
    unsigned char	c;
    struct stat		statbuf;
    long		dirtrack;
    int			tracks;
    encoding		apparent_encoding;

    for (i=256; i--; ) {
	histogram[i] = 0;
    }

    fstat(fd,&statbuf);

    /* since this is the last format to check, simply enforce the rule	*/
    /* that a jv1 must have an even number of sectors and tracks.  This	*/
    /* is evident from the file size.					*/
    /* note that it is important to check for the 18 sect tracks first	*/
    /* because a 35 track 18 sector image LOOKS like a 63 track 10 sect	*/

    if (statbuf.st_size % 4608 != 0 ) {		/* not even # of 18 sect trks	*/
	if (statbuf.st_size % 2560 != 0 ) {	/* not even # of 10 sect trks	*/
	    return(FALSE);
	} else {
	    apparent_encoding = WD_FM;		/* apparently single density	*/
	}
    } else {
	apparent_encoding = WD_MFM;	/* apparently double density	*/
    }
	
#define BYTESPERTRACK	((apparent_encoding==WD_FM)?2560:4608)

    tracks = statbuf.st_size / BYTESPERTRACK;

    dirtrack = 17; 	/* lock it in for now - may need to be	*/
			/* different for different encodings	*/

    if (lseek(fd,(off_t)(dirtrack*(long)BYTESPERTRACK),SEEK_SET) < 0)
	return(FALSE);

    for (i=BYTESPERTRACK; i--; ) {
	if (read(fd, &c, (size_t)1) < 1)
	    return(FALSE);
	histogram[c]++;
    }

    /* ok, this is goofy, but it checks to see if there is at least	*/
    /* 300 bytes of 0x00 and 0xff (together).  This convinces me that	*/
    /* there is a directory track right there.	Note that "300" isn't	*/
    /* magic...it just felt good :-).					*/

    if ( (histogram[0]+histogram[255]) < 300) {
	return(FALSE);
    }

    return(TRUE);
}


/************************************************************************
 * NAME:	jvc_report()
 *
 * DESCR:	Generates a report of the (apparently) jvc file.
 *
 * ARGS:	level refers to the amount of reporting, 1 is lowest,
 *		3 is highest.
 *
 * RETURNS:	
 *
 * NOTES:	- levels other than 1 cause a detailed look at the file
 *		  beyond the guessing that occurred before
 ************************************************************************/
int
jvc_report(int fd, struct floppy *floppy, int level)
{
    int	errors;

    printf("JVC");

    if (level == 1) {		/* first level, just indicate the type	*/
	printf("\n");
	return;
    }

    if (level == 2 || level == 3 || level == 4) {
	lseek(fd,(off_t)0,SEEK_SET);
	errors = jvc_read(fd,floppy);
	floppy_report(floppy,level,errors);
    }
}

/************************************************************************
 * NAME:	JVC format calculation Macros
 *
 * DESCR:	These macros calculate important pieces of data about a
 *		JVC file.  Note that they are all done in integer math
 *		so will truncate.  This is most important for "tracks"
 *		in that, due to padding, the track calculation may not
 *		be "even"...but the right number will come out.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- IMPORTANT - if secsize is 128 (sizecode == 0), then there
 *				MUST be an even number of sectors.  Otherwise
 *				is NOT supported by JVC.
 ************************************************************************/

#define JVC_CALC_TRACKS(filesize,headersize,sides,sectors,secsize,aflag) \
                    (((int)(filesize)) - ((int)(headersize))) / (sides) / (sectors) / ((128<<(secsize)) + ((aflag)?1:0))

#define JVC_CALC_DATASIZE(tracks,sides,sectors,secsize,aflag) \
                    (tracks) * (sides) * (sectors) * ((128<<(secsize)) + ((aflag)?1:0))

#define JVC_CALC_PADDING(tracks,sides,sectors,secsize,aflag) \
                    (256 - ((JVC_CALC_DATASIZE(tracks,sides,sectors,secsize,aflag)) % 256)) % 256


/************************************************************************
 * NAME:	jvc_get_and_check()
 *
 * DESCR:	Gets the (optional) header information from a JVC and
 *		returns it in the variables provides.
 *
 * ARGS:	
 *
 * RETURNS:	TRUE if this looks like a JVC, FALSE otherwise.
 *
 * NOTES:	- this routine will position the file pointer to the end
 *		  of any headers.
 ************************************************************************/
int
jvc_get_and_check(int	fd,
		  int	*tracks,	/* number of tracks		*/
		  int	*sides,		/* number of sides		*/
		  int	*spt,		/* sectors per track		*/
		  int	*secsize,	/* size of a sectors (in bytes)	*/
		  int	*secstart,	/* starting sector number	*/
		  int	*aflag	)	/* 0 if not using attributes	*/
{
    unsigned char	buffer[255];
    struct stat		statbuf;
    int			header_bytes;
    int			data_size;
    int			padding;

    fstat(fd,&statbuf);

    /* a jvc is either in simple form or complex form.	The simple	*/
    /* form has very obvious defaults, so it's easy to check for.	*/

    header_bytes = statbuf.st_size % 256;

    read(fd,buffer,(size_t)header_bytes);

    *spt      = (header_bytes > 0)?(int)buffer[0]  :  18;
    *sides    = (header_bytes > 1)?(int)buffer[1]  :   1;
    *secsize  = (header_bytes > 2)?(int)buffer[2]  :   1;
    *secstart = (header_bytes > 3)?(int)buffer[3]  :   1;
    *aflag    = (header_bytes > 4)?(int)buffer[4]  :   0;

    if (*spt == 0 || *sides == 0 || *secsize == 0) {
	/* failed basic test	*/
	return(FALSE);
    }

    *tracks   = JVC_CALC_TRACKS(statbuf.st_size,header_bytes,*sides,*spt,*secsize,*aflag);
    data_size = JVC_CALC_DATASIZE(*tracks,*sides,*spt,*secsize,*aflag);
    padding   = JVC_CALC_PADDING(*tracks,*sides,*spt,*secsize,*aflag);

    /* now check a few things to make sure that they look OK	*/

    if (statbuf.st_size != (header_bytes + data_size + padding)) {
	return(FALSE);	/* file size doesn't match parameters	*/
    }

    if (*sides < 1 || *sides > 2) {
	return(FALSE);	/* only can have 1 or 2 sides	*/
    }

    if (*secsize < 0 || *secsize > 3) {
	return(FALSE);	/* only a few valid secsizes	*/
    }

    if (*tracks < 20 || *tracks > 80) {	/* either bad image, or	size	*/
	return(FALSE);			/*  we can't handle.		*/
    }

    *secsize = 128 << *secsize;

    return(TRUE);
}

/************************************************************************
 * NAME:	jvc_uniform_sizecode()
 *
 * DESCR:	Checks to see if there is a uniform sizecode for all of
 *		the sectors in the image. (actually, uniformly "1")
 *
 * ARGS:	
 *
 * RETURNS:	TRUE if there is a uniform sizecode, FALSE otherwise.
 *		The value of the sizecode is written into the given place.
 *		This sizecode will be set to the last one seen if there
 *		there is not uniformity of codes.
 *
 * NOTES:	
 ************************************************************************/
int
jvc_uniform_sizecode(struct floppy *floppy, int *sizecode)
{
    int	i,j,k;

    for (i=0; i < floppy->tracks; i++) {
	for (j=0; j < floppy->sides; j++) {
	    for (k=0; k < floppy->side[j][i].sectors; k++) {
		*sizecode = floppy->side[j][i].sector[k].sizecode;
		if (*sizecode != 1) {
		    return(FALSE);
		}
	    }
	}
    }
    return(TRUE);
}

/************************************************************************
 * NAME:	jvc_uniform_firstsector()
 *
 * DESCR:	Checks to see if there is a uniform first sector number for
 *		all of the sectors in the image.
 *
 * ARGS:
 *
 * RETURNS:	Returns TRUE if there is uniform first sector number, and
 *		FALSE otherwise.  The given variable is filled in with the
 *		last first sector number seen.
 *
 *
 * NOTES:	
 ************************************************************************/
int
jvc_uniform_firstsector(struct floppy *floppy, int *firstsector)
{
    int	i,j,k;

    for (i=0; i < floppy->tracks; i++) {
	for (j=0; j < floppy->sides; j++) {

	    *firstsector = 9999;		/* set to a big "max" number	*/

	    for (k=0; k < floppy->side[j][i].sectors; k++) {
		*firstsector = MIN(floppy->side[j][i].sector[k].sector,*firstsector);
	    }

	    if (*firstsector != 1) {
		return(FALSE);
	    }
	}
    }
    return(TRUE);
}

/************************************************************************
 * NAME:	jvc_uniform_attr()
 *
 * DESCR:	Returns TRUE if the attributes are uniform for all sectors
 *		of the image, FALSE otherwise.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- uniformity means:  all marks are the same (0xfb), no
 *		  blank sectors, and no CRC errors.
 ************************************************************************/
int
jvc_uniform_attr(struct floppy *floppy)
{
    int	i,j,k;
    int	diff_mark = FALSE;
    int blank_seen = FALSE;
    int crc_error = FALSE;

    for (i=0; i < floppy->tracks; i++) {
	for (j=0; j < floppy->sides; j++) {
	    for (k=0; k < floppy->side[j][i].sectors; k++) {
		if (floppy->side[j][i].sector[k].mark != 0xfb)  {
		    diff_mark = TRUE;
		}
		if (floppy->side[j][i].sector[k].encoding == BLANK_SECTOR) {
		    blank_seen = TRUE;
		}
		if (WD_crc_body(&(floppy->side[j][i].sector[k]),TRUE)) {
		    crc_error = TRUE;
		}
	    }
	}
    }
    return(!(diff_mark || blank_seen || crc_error));
}

/************************************************************************
 * NAME:	jvc_dump_header()
 *
 * DESCR:	Used to dump a header for output of a JVC format file.
 *		This routine will check the default values and only
 *		dump a header if it is necessary.  By "necessary" I mean
 *		either the default sectors, sides, sector size, or first
 *		sector ID are different; or there is some squirrelyness
 *		in the header data for a sector somewhere.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- The data is dumped into the given buffer.  The number of
 *		  bytes dumped is returned.
 *		- it is ASSUMED that the buffer is big enough.  Note that
 *		  the definition of the JVC format limits its size to
 *		  255 bytes.
 *		- the JVC format limits sector size to a uniform size,
 *		  so the first sector is examined to determine this size.
 *		- it also limits the number of sectors to a uniform count,
 *		  so the overall sector size is used.
 *		- this routine SHOULD complain if information is lost when
 *		  converting to JVC...it doesn't
 ************************************************************************/
int
jvc_dump_header(struct floppy *floppy, unsigned char *buffer, int *diffattr)
{
    unsigned char	*ptr = buffer;
    int			 sectsizecode;
    int			 firstsector;

    *diffattr = 0;

    if (!jvc_uniform_sizecode(floppy,&sectsizecode)) {
	/* we have a non-uniform sizecode...
	   this information will be lost in JVC	*/
    }

    if (!jvc_uniform_firstsector(floppy,&firstsector)) {
	/* we have a non-uniform firstsector...
	   this info will be lost in JVC	*/
    }

    if (!jvc_uniform_attr(floppy)) {
	/* we have some non-uniform attributes...	*/
	*diffattr = 1;
    }

    /* here's a goofy brute-force approach to only	*/
    /* putting out as much header as is necessary	*/

    if (floppy->sectors != 18 || floppy->sides != 1 || sectsizecode != 1 || firstsector != 1 || *diffattr) {
	*ptr++ = floppy->sectors;
	if (floppy->sides != 1 || sectsizecode != 1 || firstsector != 1 || *diffattr) {
	    *ptr++ = floppy->sides;
	    if (sectsizecode != 1 || firstsector != 1 || *diffattr) {
		*ptr++ = sectsizecode;
		if (firstsector != 1 || *diffattr) {
		    *ptr++ = firstsector;
		    if (*diffattr) {
			*ptr++ = *diffattr;
		    }
		}
	    }
	}
    }

    return(ptr-buffer);
}

/************************************************************************
 * NAME:	jvc_dump_padding()
 *
 * DESCR:	Dumps padding for the jvc file.  Turns out only to be
 *		necessary if attributes are used.
 *
 * ARGS:	
 *
 * RETURNS:	number of bytes added on
 *
 * NOTES:	- odd numbers of 128-byte sectors are NOT supported
 ************************************************************************/
int
jvc_dump_padding(struct floppy *floppy, unsigned char *buffer)
{
    unsigned char	*ptr = buffer;
    int			 count;
    int			 sectsizecode;
    int			 diffattr = 0;

    if (!jvc_uniform_sizecode(floppy,&sectsizecode)) {
	/* we have a non-uniform sizecode...
	   this information will be lost in JVC	*/
    }

    if (!jvc_uniform_attr(floppy)) {
	/* we have some non-uniform attributes...	*/
	diffattr = 1;
    }

    count = JVC_CALC_PADDING(floppy->tracks, floppy->sides, floppy->sectors, sectsizecode, diffattr);

    for ( ; count--; ) {
	*ptr++ = '\0';
    }

    return(ptr-buffer);
}


/************************************************************************
 * NAME:	jvc_guesser()
 *
 * DESCR:	Tries to guess a jvc format.  This format is either simple
 *		or complex.  The simple format is almost exactly like jv1 DD
 *		but sets up different things on the image.  The more complex
 *		format has things such as skipped sectors, etc.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- This should be guessed BEFORE jv1, because jv1 may pick
 *		  up the simple form of jvc.
 *		- The simple form defaults to 18 track, DD, 1 side
 ************************************************************************/
int
jvc_guesser(int fd)
{
    unsigned char	buffer[255];
    struct stat		statbuf;
    int			header_bytes;
    int			tracks;
    int			spt;			/* sectors per track	*/
    int			sides;
    int			secstart;		/* starting sec number	*/
    int			extra_byte;		/* TRUE -> extra byte	*/
    int			secsize;		/* bytes in a sectors	*/

    int			ok;

    ok = jvc_get_and_check(fd,
			   &tracks,
			   &sides,
			   &spt,
			   &secsize,
			   &secstart,
			   &extra_byte);
    return(ok);
}

/************************************************************************
 * NAME:	jvc_read()
 *
 * DESCR:	Tries to read a jvc format.  This format is either simple
 *		or hard.  The simple format is almost exactly like jv1 DD
 *		but sets up different things on the image.  The more complex
 *		format has things such as skipped sectors, etc.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- This should be guessed BEFORE jv1, because jv1 will pick
 *		  up the simple form of jvc.
 *		- The simple form defaults to 18 track, DD, 1 side
 ************************************************************************/
int
jvc_read(int fd, struct floppy *floppy)
{
    struct stat		statbuf;
    int			header_bytes;
    int			tracks;
    int			spt;			/* sectors per track	*/
    int			sides;
    int			secstart;		/* starting sec number	*/
    int			extra_byte;		/* TRUE -> extra byte	*/
    int			secsize;		/* bytes in a sectors	*/

    int			ok;

    char		flag;
    int			track;
    int			side;
    int			sector;

    ok = jvc_get_and_check(fd,
			   &tracks,
			   &sides,
			   &spt,
			   &secsize,
			   &secstart,
			   &extra_byte);

    if (!ok) {
	return(FALSE);
    }

    floppy->tracks = tracks;
    floppy->sectors = spt;
    floppy->sides = sides;

    floppy->encoding = WD_MFM;
    floppy->write_protect = 0;		/* not write-protected		*/

    floppy->side[0] = (struct track *)malloc(floppy->tracks*sizeof(struct track));
    if (floppy->sides > 1) {
	floppy->side[1] = (struct track *)malloc(floppy->tracks*sizeof(struct track));
    }

    for (track = 0; track < floppy->tracks; track++) {
	for (side = 0; side < floppy->sides; side++) {
	    floppy->side[side][track].sector = 
		(struct sector *)malloc(floppy->sectors*sizeof(struct sector));
	    floppy->side[side][track].sectors = floppy->sectors;
	}
    }

    for(track =0; track < floppy->tracks; track++) {
	int	count;

	for (side = 0; side < floppy->sides; side++) {

	    for (sector = 0; sector < floppy->side[side][track].sectors; sector++) {

		if (extra_byte) {
		    read(fd,&flag,1);
		}

		/* NOTE - this is broken in that the data buffer is ONLY 256	*/
		/*	  but the secsize can be bigger!			*/

		count = read(fd,floppy->side[side][track].sector[sector].data,secsize);

		if (count < secsize) {
		    break;	/* must be done, didn't get a complete track*/
		}

		floppy->side[side][track].sector[sector].id = track;
		floppy->side[side][track].sector[sector].side = 0;
		floppy->side[side][track].sector[sector].sector = sector + secstart;
		floppy->side[side][track].sector[sector].sizecode = 1;
		floppy->side[side][track].sector[sector].size = secsize;

		floppy->side[side][track].sector[sector].mark = \
		    (extra_byte && (flag & JVC_ATTR_DELETED))?0xf8:0xfb;

		floppy->side[side][track].sector[sector].encoding = 
		    (extra_byte && (flag & JVC_ATTR_SECTERR))?BLANK_SECTOR:floppy->encoding;

		/* do the CRCs AFTER the rest of the header was done! */

		(void)WD_crc_header(&(floppy->side[side][track].sector[sector]),0);
		(void)WD_crc_body(&(floppy->side[side][track].sector[sector]),0);

		floppy->side[side][track].sector[sector].dataCRC += \
		    (extra_byte && (flag & JVC_ATTR_CRCERR))?1:0;	/* screw up CRC if indicated */
	    }

	    if (count < secsize)
		break; 	/* must be done	*/
	}
    }

    floppy->tracks = track;

#ifdef NOTDEF
    /* put directory marks in the directory track	*/

    for (sector = 0; sector < floppy->sectors; sector++) {
	floppy->side[0][dirtrack].sector[sector].mark = (floppy->encoding==WD_FM)?0xfa:0xf8;
    }
#endif
    /* need to do real errors	*/

    return(0);

}

/************************************************************************
 * NAME:	jvc_dump_attribute()
 *
 * DESCR:	Fills in a little buffer with a single byte that
 *		contains the attributes as they should be written into
 *		a JVC format for a sector.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 *	    bit 0, 1, 2 - not used
 *	    bit 3 - set on sector CRC error
 *	    bit 4 - set if sector not found
 *	    bit 5 - 1 = deleted data mark, 0 = data mark
 *	    bit 6, 7 - not used
 *
 *		- bit 4 is set if the given sector is a blank
 ************************************************************************/
void
jvc_dump_attribute(struct sector *sector, unsigned char *buffer)
{
    *buffer = 0;

    if (WD_crc_body(sector,TRUE)) {	/* returns 0 upon GOOD CRC	*/
	*buffer |= JVC_ATTR_CRCERR;
    }

    if (sector->encoding == BLANK_SECTOR) {
	*buffer |= JVC_ATTR_SECTERR;
    }

    if (sector->mark != 0xfb) {	/* something other means deleted */
	*buffer |= JVC_ATTR_DELETED;
    }
}

/************************************************************************
 * NAME:	jvc_dump()
 *
 * DESCR:	Dumps out the floppy in jvc format.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- since the actual id of the sectors is not stored in JVC,
 *		  the sectors need to be dumped out in order.
 *		- if there are weird sector numbers that don't start with
 *		  with the "first sector" they will be lost.
 ************************************************************************/
int
jvc_dump(int fd, struct floppy *floppy)
{
    unsigned char	buffer[255];
    int			diffattr;
    int			hdrbytes;
    int			i,j,k;
    int			firstsector;

    hdrbytes = jvc_dump_header(floppy,buffer,&diffattr);

    (void)jvc_uniform_firstsector(floppy,&firstsector);	/* ignore error conds */

    if (hdrbytes) {
	write(fd,buffer,hdrbytes);
    }

    /* Now dump the sector data...the only weirdness is if the diffattr is TRUE,
       then each sector needs to be preceeded by a byte indicating the attributes
       for that sector */

    for (i=0; i < floppy->tracks; i++) {
	for (j=0; j < floppy->sides; j++) {
	    for (k=0; k < floppy->side[j][i].sectors; k++) {
		if (diffattr) {
		    jvc_dump_attribute(&(LogicalSector(floppy,j,i,k+firstsector)),buffer);
		    write(fd,buffer,(size_t)1);
		}
		write(fd,LogicalSector(floppy,j,i,k+firstsector).data,(size_t)256);
	    }
	}
    }

    {
	int padding = jvc_dump_padding(floppy, buffer);

	if (padding) {
	    write(fd,buffer,(size_t)padding);
	}
    }
    
							      
}

/************************************************************************
 * NAME:	jv3_report()
 *
 * DESCR:	Generates a report of the (apparently) jv3 file.
 *
 * ARGS:	level refers to the amount of reporting, 1 is lowest,
 *		3 is highest.
 *
 * RETURNS:	
 *
 * NOTES:	- levels other than 1 cause a detailed look at the file
 *		  beyond the guessing that occurred before
 ************************************************************************/
int
jv3_report(int fd, struct floppy *floppy, int level)
{
    int	errors;

    printf("JV3");

    if (level == 1) {		/* first level, just indicate the type	*/
	printf("\n");
	return;
    }

    if (level == 2 || level == 3 || level == 4) {
	lseek(fd,(off_t)0,SEEK_SET);
	errors = jv3_read(fd,floppy);
	floppy_report(floppy,level,errors);
    }
}

int jv3_guesser(int fd)
{
    int		header, data, sectors;
    struct stat	statbuf;

    fstat(fd,&statbuf);

    /* ** Mon Apr 11 14:28:49 2005 **					*/
    /* this routine was far too liberal before.  It was easily fooled	*/
    /* by images with all 0xff's.  So it has been tightened up to work	*/
    /* ONLY with JV3 images that have the correct size, which is:	*/
    /*    HEADERS = 2901 * 3 + 1  (for the JVC header)			*/
    /*    DATA = exact multiple of 35/40/80 tracks of 256-byte sectors	*/
    /*		(this should catch DD and SD)				*/
    /* This assumes that the JVC header is 3 bytes times 2901 plus WP.	*/

    header = 2901 * 3 + 1;
    data = statbuf.st_size - header;

    if (!EVEN_MULTIPLE(data,256)) {
	return(FALSE);
    }

    sectors = data/256;

    if (EVEN_MULTIPLE(sectors,10)) {
	switch(sectors/10) {
            case 35:   
            case 40:
            case 80:
		return(TRUE);		/* looks like a good JV3	*/
	}
    }

    if (EVEN_MULTIPLE(sectors,18)) {
	switch(sectors/18) {
            case 35:   
            case 40:
            case 80:
		return(TRUE);		/* looks like a good JV3	*/
	}
    }

    return(FALSE);

    /* good enough for now	*/
    /* i can imagine in the future checking the sector size & mark	*/
}

/************************************************************************
 * NAME:	jv1_read()
 *
 * DESCR:	Read in the jv1 format.  Pretty simple...just a bunch
 *		of tracks.  Note that it is assumed that jv1 is SSSD
 *		only.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int jv1_read(int fd, struct floppy *floppy)
{
    int		track;
    int 	track_length = 256;	/* should move this to floppy	*/
    int		dirtrack;
    int		sector;
    struct stat	statbuf;

    fstat(fd,&statbuf);

    if (statbuf.st_size % 4608 == 0 ) {
	floppy->encoding = WD_MFM;		/* apparently double density	*/
    } else if (statbuf.st_size % 2560 == 0 ) {
	floppy->encoding = WD_FM;		/* apparently single density	*/
    } else {
	floppy->encoding = UNKNOWN_ENCODING;
    }

    floppy->write_protect = 0;		/* not write-protected		*/
    floppy->sides = 1;
    floppy->sectors = (floppy->encoding==WD_FM)?10:18;
    floppy->tracks = statbuf.st_size / ((floppy->encoding==WD_FM)?2560:4608);

    floppy->side[0] = (struct track *)malloc(floppy->tracks*sizeof(struct track));

    for (track = 0; track < floppy->tracks; track++) {
	floppy->side[0][track].sector = 
	    (struct sector *)malloc(floppy->sectors*sizeof(struct sector));
	floppy->side[0][track].sectors = floppy->sectors;
    }

    for(track =0; track < floppy->tracks; track++) {
	int	count;

	for (sector = 0; sector < floppy->sectors; sector++) {

	    count = read(fd,floppy->side[0][track].sector[sector].data,(size_t)track_length);

	    if (count < track_length) {
		break;	/* must be done, didn't get a complete track*/
	    }

	    floppy->side[0][track].sector[sector].id = track;
	    floppy->side[0][track].sector[sector].side = 0;
	    floppy->side[0][track].sector[sector].sector = sector;
	    floppy->side[0][track].sector[sector].sizecode = 1;
	    floppy->side[0][track].sector[sector].size = 256;
	    floppy->side[0][track].sector[sector].mark = 0xfb;
	    floppy->side[0][track].sector[sector].headCRC = 0;
	    floppy->side[0][track].sector[sector].dataCRC = 0;
	    floppy->side[0][track].sector[sector].encoding = floppy->encoding;
	}

	if (count < track_length)
	    break; 	/* must be done	*/
    }

    floppy->tracks = track;

    dirtrack = 17; 	/* lock it in for now - may need to be	*/
			/* different for different encodings	*/

    /* put directory marks in the directory track	*/

    for (sector = 0; sector < floppy->sectors; sector++) {
	floppy->side[0][dirtrack].sector[sector].mark = (floppy->encoding==WD_FM)?0xfa:0xf8;
    }

    /* need to do real errors	*/

    floppy_crc_set(floppy);	/* set the CRCs	*/

    return(0);
}

/************************************************************************
 * NAME:	jv1_dump()
 *
 * DESCR:	Write to the jv1 format.  Pretty simple here too...note
 *		that a ton of information is lost when going to jv1.  It
 *		only handles non-copy-protected disks, single sided.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	ERRORS are not checked.  Booooo,  hissss.
 ************************************************************************/
jv1_dump(int fd,struct floppy *floppy )
{
    int	track;
    int sector;

    for (track = 0; track < floppy->tracks; track++) {
	for (sector = 0; sector < floppy->sectors; sector++) {
	    write(fd,LogicalSector(floppy,0,track,sector).data,256);
	}
    }
}


/* NEED BETTER ERRORS HERE */

ErrorStatus
jv3_read(int fd, struct floppy *floppy)
{
    int		i, j;
    long	data_size;
    struct jv3	jv3_buffer;
    struct stat	statbuf;
    int		trackmap[2][256];	/* 256 possible different tracks
					   with 2 sides (counts the
					   number of sectors too	*/
    int		revtrackmap[2][256];	/* reverse index of the tracmap	*/

    int		sectormap[2][256];	/* sector count			*/

    /* NOTE - ZAXXON.dsk (DMK format) can't be converted to JV3 right	*/
    /*		because track 2 is repeated on that image.  There's no	*/
    /*		way to represent this in jv3 that I can figure...	*/

    for (i=2; i--; ) {		/* initialize track & sector maps	*/
	for (j=256; j--; ) {
	    trackmap[i][j] = 0;
	    revtrackmap[i][j] = 0;
	    sectormap[i][j] = 0;
	}
    }

    fstat(fd,&statbuf);

    /* NOTE: +1 is for reading the write-protect too			*/

    i = read(fd,&jv3_buffer,sizeof(struct jv3_header)*2901+1);

    if (i < sizeof(struct jv3_header)*2901+1) {
	return(E_FATAL | E_JV3_BAD_FORMAT);
    }

    if (jv3_buffer.write_protect != 0x00 && jv3_buffer.write_protect != 0xff) {
	return(E_FATAL | E_JV3_BAD_FORMAT);
    }

    /* looks like we got enough data for the headers, now read data	*/

    data_size = 0;	    /* since data_size is a long, if wild #'s
			       are in the headers (ie - bad format)
			       then it won't roll-over and look small	*/

    /* this is just a size calculation pass */

    for (i=0; i < 2901; i++) {
	if (!JV3_IS_FREE_SEC(jv3_buffer.headers[i])) {
	    data_size += JV3_SIZE(jv3_buffer.headers[i]);
	    trackmap[JV3_SIDE(jv3_buffer.headers[i])][jv3_buffer.headers[i].track] += 1;
	    sectormap[JV3_SIDE(jv3_buffer.headers[i])][jv3_buffer.headers[i].track] += 1;
/******
	    fprintf(stderr,"track %d, side %d, sector %d, density %s, dam %0x, size %ld\n",
		    jv3_buffer.headers[i].track,
		    JV3_SIDE(jv3_buffer.headers[i]),
		    jv3_buffer.headers[i].sector,
		    JV3_IS_DD(jv3_buffer.headers[i])?"double":"single",
		    JV3_DAM(jv3_buffer.headers[i]),
		    data_size);
*******/
	}
    }

    /* current position in the file is sizeof(jv3_header)*2901+1	*/
    /* I don't remember the system call to find position...so I'm just	*/
    /* going to use this for now					*/

    if (sizeof(struct jv3_header)*2901 + 1 + data_size > statbuf.st_size) {
	return(E_FATAL|E_JV3_BAD_FORMAT);
    }

    /* Looks like a good format, now go through and get it into a	*/
    /* floppy structure							*/

    floppy->write_protect = jv3_buffer.write_protect; /* this right?	*/
    floppy->sides = 1;
    floppy->tracks = 0;
    floppy->encoding = WD_FM;		/* just for basic indication	*/

    /* figure out how many distinct tracks there are and allocate	*/
    /* NOTE - assumes the same number of tracks both sides		*/
    /*		and that side 0 is NEVER clear (relative to side 1)	*/
   
    {
	/* update the track map with logical track numbers	*/

	int	track = 0;

	for (i=0; i < 256; i++ ) {
	    if (trackmap[0][i] !=0) {
		floppy->tracks++;
		trackmap[0][i] = track;
		revtrackmap[0][track] = i;
		track++;
	    }
	}
    }

    {
	/* now see if there are tracks on side 1 and do that map	*/

	int	track = 0;
	int	tracks = 0;

	floppy->sides = 1;

	for (i=0; i<256; i++ ) {
	    if (trackmap[1][i] !=0) {
		tracks += 1;
		floppy->sides = 2;
		trackmap[1][i] = track;
		revtrackmap[1][track] = i;
		track++;
	    }
	}

	/* see if the number of tracks on each side is consistent	*/

	if (floppy->sides == 2 && floppy->tracks != tracks) {
	    return(E_FATAL|E_JV3_SIDE_INCON);
	}
    }

    /* allocate the track storage	*/

    floppy->side[0] = (struct track *)malloc(floppy->tracks*sizeof(struct track));
    if (floppy->sides == 2)
	floppy->side[1] = (struct track *)malloc(floppy->tracks*sizeof(struct track));

    /* allocate the sectors in each track	*/
    
    for (i = 0; i < floppy->sides; i++) {
	for (j = 0; j < floppy->tracks; j++) {
	    floppy->side[i][j].sectors = sectormap[i][revtrackmap[i][j]];
	    floppy->side[i][j].sector = (struct sector *)
	                         malloc(floppy->side[i][j].sectors*sizeof(struct sector));
	    /* clear the number of sectors, so it can be used during load	*/
	    floppy->side[i][j].sectors = 0;
		    
	}
    }

    {
	for (i=0; i < 2901; i++) {

#define THIS_ENTRY	jv3_buffer.headers[i]

	    if (!JV3_IS_FREE_SEC(THIS_ENTRY)) {

#define MAPPED_TRACK	trackmap[JV3_SIDE(THIS_ENTRY)][THIS_ENTRY.track]
#define THIS_TRACK	floppy->side[JV3_SIDE(THIS_ENTRY)][MAPPED_TRACK]
#define THIS_SECTOR	THIS_TRACK.sector[THIS_TRACK.sectors]

		THIS_SECTOR.id = THIS_ENTRY.track;
		THIS_SECTOR.side = JV3_SIDE(THIS_ENTRY);
		THIS_SECTOR.sector = THIS_ENTRY.sector;

		THIS_SECTOR.sizecode = JV3_SIZE_CODE(THIS_ENTRY);
		THIS_SECTOR.size = JV3_SIZE(THIS_ENTRY);
		THIS_SECTOR.mark = JV3_DAM(THIS_ENTRY);

		/* NOTE - need to fix these three *******/

		THIS_SECTOR.headCRC = 0;
		THIS_SECTOR.dataCRC = 0;

		/****************************************/

		THIS_SECTOR.encoding = JV3_IS_DD(THIS_ENTRY)?WD_MFM:WD_FM;

		if (THIS_SECTOR.size > 256) {
		    /* can't support sizes other than 256 right now */
		    return(E_FATAL|E_JV3_DATA_SIZE);
		}

		read(fd,THIS_SECTOR.data,(size_t)THIS_SECTOR.size);

		THIS_TRACK.sectors++;
	    }
	}
    }

    /* update the maximum number of sectors	*/

    floppy->sectors = 0;
    for (i=0; i < floppy->sides; i++) {
	for (j=0; j < floppy->tracks; j++) {
	    floppy->sectors = MAX(floppy->sectors,floppy->side[i][j].sectors);
	}
    }

    {
	int mixed;

	floppy->encoding = floppy_overall_encoding(floppy, &mixed, -1);
    }

    /* TODO - there is an optional second data set...should read it too	*/

    floppy_crc_set(floppy);	/* set the CRCs	*/

    return(E_NONE);
}

/************************************************************************
 * NAME:	jv3_dump()
 *
 * DESCR:	Dumps out the floppy in jv3 format.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
jv3_dump(int fd, struct floppy *floppy)
{
}


/************************************************************************
 * NAME:	jv1_init()
 *
 * DESCR:	Registers the jv1 floppy format.
 *
 * ARGS:	
 *
 * RETURNS:	TRUE if the registration went OK, FALSE otherwise.
 *
 * NOTES:	
 ************************************************************************/
int
jv1_init()
{
    if (!floppy_format_register("JV1", "TRS-80 JV1",jv1_guesser, jv1_read, jv1_dump, jv1_report)) {
	return(FALSE);
    }

    if (!floppy_fileext_register("jv1", EXT_NOT_SPECIFIC, "JV1")) {
	return(FALSE);
    }

    if (!floppy_fileext_register("dsk", EXT_NOT_SPECIFIC, "JV1")) {
	return(FALSE);
    }

    return(TRUE);
}

/************************************************************************
 * NAME:	jv3_init()
 *
 * DESCR:	Registers the jv3 floppy format.
 *
 * ARGS:	
 *
 * RETURNS:	TRUE if the registration went OK, FALSE otherwise.
 *
 * NOTES:	- there is no WRITE for jv3 at this time.
 ************************************************************************/
int
jv3_init()
{
    if (!floppy_format_register("JV3", "TRS-80 JV1", jv3_guesser, jv3_read, NULL, jv3_report)) {
	return(FALSE);
    }

    if (!floppy_fileext_register("jv3", EXT_SOMEWHAT_SPECIFIC, "JV3")) {
	return(FALSE);
    }

    if (!floppy_fileext_register("dsk", EXT_SOMEWHAT_SPECIFIC, "JV3")) {
	return(FALSE);
    }

    return(TRUE);
}

/************************************************************************
 * NAME:	jvc_init()
 *
 * DESCR:	Registers the jvc floppy format.
 *
 * ARGS:	
 *
 * RETURNS:	TRUE if the registration went OK, FALSE otherwise.
 *
 * NOTES:	- there is no WRITE for jvc at this time.
 ************************************************************************/
int
jvc_init()
{
    if(!floppy_format_register("JVC", "TRS-80 CoCo JVC",jvc_guesser, jvc_read, jvc_dump, jvc_report)) {
	return(FALSE);
    }

    if (!floppy_fileext_register("jvc", EXT_SOMEWHAT_SPECIFIC, "JVC")) {
	return(FALSE);
    }

    if (!floppy_fileext_register("dsk", EXT_SOMEWHAT_SPECIFIC, "JVC")) {
	return(FALSE);
    }

    return(TRUE);
}
